﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels.DiscountManagement;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/discount-management")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class DiscountManagementController : BaseController
    {
        /// <summary>
        /// The discount management services.
        /// </summary>
        private readonly IDiscountManagementService discountManagementService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public DiscountManagementController(IDiscountManagementService discountManagementService, IAuditLogService auditLogServices)
        {
            this.discountManagementService = discountManagementService;
            this.auditLogServices = auditLogServices;
        }
        /// <summary>
        /// The modify discount modules async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPost]
        [Route("modify-discount-module")]
        public async Task<ActionResult> ModifyDiscountModulesAsync([FromBody] DiscountModuleModel model, [FromHeader] LocationHeader header)
        {
            model = (DiscountModuleModel)EmptyFilter.Handler(model);
            var response = model.DiscountModuleId > 0 ? await this.discountManagementService.UpdateDiscountModule(model) : await this.discountManagementService.AddDiscountModule(model);
            var addEdit = model.DiscountModuleId > 0 ? "updated" : "added";
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.DiscountManagement,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" <b>{model.CreatedByName}</b> has <b>{addEdit}</b> Discount Management <b>{model.ModuleName}</b> on {DateTime.UtcNow.AddMinutes(330)}",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(response);
        }
        /// <summary>
        /// The fetch discount modules async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPost]
        [Route("fetch-discount-module")]
        public async Task<ActionResult> FetchDiscountModulesAsync([FromBody] DiscountModuleModel model)
        {
            model ??= new DiscountModuleModel();
            var response = await this.discountManagementService.FetchAllDiscountModulesAsync(model);
            return this.Success(response);
        }
        /// <summary>
        /// The modify discounts per modules async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPost]
        [Route("modify-discount-per-module")]
        public async Task<ActionResult> ModifyDiscountsPerModuleAsync([FromBody] DiscountsPerModuleModel model)
        {
            model = (DiscountsPerModuleModel)EmptyFilter.Handler(model);
            var response = model.DiscountsPerModuleId > 0 ? await this.discountManagementService.UpdateDiscountsPerModuleAsync(model) :
                await this.discountManagementService.AddDiscountsPerModuleAsync(model);
            return this.Success(response);
        }
        /// <summary>
        /// The get modules detail async.
        /// </summary>
        /// <param name="moduleName">
        /// The module name.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("module-detail")]
        public async Task<ActionResult> GetModulesDetailAsync(string moduleName)
        {
            if (string.IsNullOrEmpty(moduleName))
            {
                return this.BadRequest("Invalid parameters");
            }
            var response = await this.discountManagementService.GetModuleDetailsAsync(moduleName);
            return this.Success(response);
        }
        /// <summary>
        /// The modify discounts per modules async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPost]
        [Route("get-role-based-discount")]
        public async Task<ActionResult> GetRoleBasedDiscountAsync([FromBody] DiscountsPerModuleModel model)
        {
            model = (DiscountsPerModuleModel)EmptyFilter.Handler(model);
            var response = await this.discountManagementService.GetRoleBasedDisocuntOnModule(model);
            return this.Success(response);
        }

        /// <summary>
        /// To avtivate or deactivate role.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <param name="header">
        /// The header.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Specialization deleted successfully.
        /// - 409 - Specialization can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("discount-Management-modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] DiscountModuleModel model, [FromHeader] LocationHeader header)
        {
            model = (DiscountModuleModel)EmptyFilter.Handler(model);
            var response = await this.discountManagementService.ActivateOrDeactivate(model);
            if (response == 0)
            {
                return this.ServerError();
            }
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.DiscountManagement,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" <b>{model.ModifiedByName}</b> has <b>{((bool)model.Active ? "Activated" : "Deactivated")}</b> Discount Management <b>{model.ModuleName}</b> on {DateTime.UtcNow.AddMinutes(330)}",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(response);

        }
    }
}